module cpu
(
    // --- УПРАВЛЯЮЩИЕ
    input               clock,      // 25Мгц
    input               rst_n,      // =0 Сброс процессора
    input               ce,         // =1 Активация чипа
    output              m0,         // Первый такт
    // --- ПАМЯТЬ
    output      [19:0]  a,          // Адрес в общей памяти
    input       [ 7:0]  i,          // Данные из памяти
    output reg  [ 7:0]  o,          // Данные в память
    output reg          w,          // Запись в память,
    // --- РЕГИСТРЫ
    output reg  [15:0]  ax,
    output reg  [15:0]  bx,
    output reg  [15:0]  cx,
    output reg  [15:0]  dx,
    output reg  [15:0]  sp,
    output reg  [15:0]  bp,
    output reg  [15:0]  si,
    output reg  [15:0]  di,
    // --- СЕГМЕНТЫ
    output reg  [15:0]  es,
    output reg  [15:0]  cs,
    output reg  [15:0]  ss,
    output reg  [15:0]  ds,
    // --- УПРАВЛЯЮЩИЕ
    output reg  [15:0]  ip,
    output reg  [11:0]  flags,
    // --- ИНИЦИАЛИЗАЦИЯ
    input       [15:0]  _ax,
    input       [15:0]  _bx,
    input       [15:0]  _cx,
    input       [15:0]  _dx,
    input       [15:0]  _sp,
    input       [15:0]  _bp,
    input       [15:0]  _si,
    input       [15:0]  _di,
    input       [15:0]  _es,
    input       [15:0]  _cs,
    input       [15:0]  _ss,
    input       [15:0]  _ds,
    input       [15:0]  _ip,
    input       [11:0]  _flags
);

assign a = cp ? sgn*16 + ea : cs*16 + ip;

localparam
    RUN         = 0, // Исполнение
    MODRM       = 1, // Разбор операндов
    WB          = 2, // Запись результата
    PUSH        = 3, // Запись wb в стек
    POP         = 4, // Чтение из стека -> wb
    DIV         = 5, // Деление op1/op2
    INTERRUPT   = 6, // Вызов прерывания
    UNDEF       = 7; // Ошибка опкода

localparam ADD = 0, OR  = 1, ADC = 2, SBB  = 3, AND = 4, SUB = 5, XOR = 6, CMP = 7;
localparam ROL = 0, ROR = 1, RCL = 2, RCR  = 3, SHL = 4, SHR = 5, SAL = 6, SAR = 7;
localparam CF  = 0, PF  = 2, AF  = 4, ZF   = 6, SF  = 7, TF  = 8, IF  = 9, DF  = 10, OF = 11;

// -----------------------------------------------------
`define CPEN cp <= cpen; if (!cpen) begin m1 <= 0; t <= RUN; end
`define TERM {m, rep, over} <= 1'b0;
// -----------------------------------------------------
reg         cp;                     // =1 Указатель на SGN:EA =0 Иначе CS:IP
reg         cpm;                    // =0 Устанавливается cp после MODRM
reg         size;                   // =1 16bit =0 8bit
reg         dir;                    // =0 rm,r; =1 r,rm
reg         cpen;                   // =0 То пропускает чтение операндов
reg         over;                   // =1 Сегмент переопределен
reg         intrc;                  // Предыдущее значение intr
reg [ 1:0]  rep;                    // Наличие REP:
reg [ 3:0]  t, next;                // Исполняемая команда (t) в данный момент
reg [ 3:0]  m, m1;                  // Фаза исполнения T (m), MODRM и субфаза
reg [ 2:0]  m2, m3, m4;             // Фаза исполнения WB, PUSH/POP
reg [ 2:0]  alu;                    // Функция АЛУ или сдвигов
reg [ 7:0]  modrm, opcache;         // Кеш modrm и opcache
reg [ 7:0]  interrupt;              // Номер прерывания
reg [15:0]  sgn, ea;                // Выбранный SEGMENT:EA
reg [15:0]  op1, op2, wb, t16;      // Операнды; wb-что записывать
// -----------------------------------------------------
wire [7:0]  opcode = m || t ? opcache : i;
wire        m0 = (t == RUN && m == 0);
wire        c0 = m0 && {over, rep} == 3'b0;
// -----------------------------------------------------
// Мультиплексор на выбор регистров из диапазона [2:0]
wire [15:0] i20 =
    i[2:0] == 3'h0 ? (size ? ax : ax[ 7:0]) :
    i[2:0] == 3'h1 ? (size ? cx : cx[ 7:0]) :
    i[2:0] == 3'h2 ? (size ? dx : dx[ 7:0]) :
    i[2:0] == 3'h3 ? (size ? bx : bx[ 7:0]) :
    i[2:0] == 3'h4 ? (size ? sp : ax[15:8]) :
    i[2:0] == 3'h5 ? (size ? bp : cx[15:8]) :
    i[2:0] == 3'h6 ? (size ? si : dx[15:8]) :
                     (size ? di : bx[15:8]);

// Мультиплексор на выбор регистров из диапазона [5:3]
wire [15:0] i53 =
    i[5:3] == 3'h0 ? (size ? ax : ax[ 7:0]) :
    i[5:3] == 3'h1 ? (size ? cx : cx[ 7:0]) :
    i[5:3] == 3'h2 ? (size ? dx : dx[ 7:0]) :
    i[5:3] == 3'h3 ? (size ? bx : bx[ 7:0]) :
    i[5:3] == 3'h4 ? (size ? sp : ax[15:8]) :
    i[5:3] == 3'h5 ? (size ? bp : cx[15:8]) :
    i[5:3] == 3'h6 ? (size ? si : dx[15:8]) :
                     (size ? di : bx[15:8]);

wire [15:0] sign = {{8{i[7]}}, i};
// -----------------------------------------------------
wire [15:0] ar = 1'b0;
wire [11:0] af = 1'b0;
// -----------------------------------------------------

always @(posedge clock)
// Сброс процессора
if (rst_n == 0) begin

    t  <= RUN;
    m  <= 0;
    cp <= 0;

    // Копирование первоначальных значений при сбросе
    ax <= _ax; bx <= _bx; cx <= _cx; dx <= _dx;
    sp <= _sp; bp <= _bp; si <= _si; di <= _di;
    es <= _es; cs <= _cs; ss <= _ss; ds <= _ds;
    ip <= _ip; flags <= _flags;

// Запуск выполнения команд процессора
end
else if (ce) begin

    w <= 0;

    // Очистить управляющие регистры в начале инструкции
    if (m0) begin

        next    <= RUN;
        ip      <= ip + 1;
        m       <= 1;
        m1      <= 0;
        m2      <= 0;
        m3      <= 0;
        m4      <= 0;
        cpm     <= 1;
        cpen    <= 1;
        opcache <= i;
        dir     <= i[1];
        size    <= i[0];

        if (c0) sgn <= ds;

    end

    case (t)

    // ИСПОЛНЕНИЕ ИНСТРУКЦИИ
    RUN: casex (opcode)

        8'b00xxx0xx: case (m) // ### ALU-операции с операндами ModRM [3T+]

            0: begin t <= MODRM; alu <= opcode[5:3]; end
            1: begin t <= alu == CMP ? RUN : WB; wb <= ar; flags <= af; `TERM end

        endcase

        8'b001xx110: case (m) // ### Префикс ES/CS/SS/DS [1T]

            0: begin

                m    <= 0; // Продолжим считывать префиксы или опкод
                over <= 1; // Установим отметку, что был сегментный префикс

                // Выбираем новый сегмент
                case (opcode[4:3])
                0: sgn <= es;
                1: sgn <= cs;
                2: sgn <= ss;
                3: sgn <= ds;
                endcase

            end

        endcase

        8'b1111001x: case (m) // ### REPNZ, REPZ [1T]

            0: begin m <= 0; rep <= i[1:0]; end

        endcase

    endcase

    // СЧИТЫВАНИЕ ОПЕРАНДОВ ИЗ РЕГИСТРОВ ИЛИ ИЗ ПАМЯТИ
    MODRM: case (m1)

        0: begin

            modrm   <= i;
            ip      <= ip + 1;
            op1     <= dir ? i53 : i20;
            op2     <= dir ? i20 : i53;

            case (i[2:0])
            3'b000: ea <= bx + si;
            3'b001: ea <= bx + di;
            3'b010: ea <= bp + si;
            3'b011: ea <= bp + di;
            3'b100: ea <= si;
            3'b101: ea <= di;
            3'b110: ea <= i[7:6] ? bp : 0;
            3'b111: ea <= bx;
            endcase

            casex (i)
            8'b00_xxx_110: begin m1 <= 2; end           // OFFSET16
            8'b00_xxx_xxx: begin m1 <= 4; `CPEN; end    // Читать операнд
            8'b01_xxx_xxx: begin m1 <= 1; end           // +Смещение 8 бит
            8'b10_xxx_xxx: begin m1 <= 2; end           // +Смещение 16 бит
            8'b11_xxx_xxx: begin m1 <= 0; t <= RUN; end // Регистры. Вернуться к RUN
            endcase

            if (!over && ((^i[7:6] && i[2:0] == 3'b110) || i[2:1] == 2'b01)) sgn <= ss;

        end

        // DISP8/16
        1: begin m1 <= 4; ip <= ip + 1; ea <= ea + sign;       `CPEN; end
        2: begin m1 <= 3; ip <= ip + 1; ea <= ea + {8'h00, i}; end
        3: begin m1 <= 4; ip <= ip + 1; ea <= ea + {i, 8'h00}; `CPEN; end

        // Читать 8-битный операнд
        4: begin

            if (dir) op2 <= i; else op1 <= i;
            if (size) begin m1 <= 5; ea <= ea + 1; end
            else      begin m1 <= 0; cp <= cpm; t <= RUN; end

        end

        // Читать 16-битный операнд
        5: begin

            if (dir) op2[15:8] <= i; else op1[15:8] <= i;

            t  <= RUN;
            m1 <= 0;
            ea <= ea - 1;
            cp <= cpm;

        end

    endcase

    // [1T,2-3T] ЗАПИСЬ РЕЗУЛЬТАТОВ WB,DIR,SIZE,MODRM В ПАМЯТЬ/РЕГИСТРЫ
    WB: case (m2)

        // Записать в регистры, если это явно указано
        0: if (dir || modrm[7:6] == 2'b11) begin

            case (dir ? modrm[5:3] : modrm[2:0])
            0: if (size) ax <= wb; else ax[ 7:0] <= wb[7:0];
            1: if (size) cx <= wb; else cx[ 7:0] <= wb[7:0];
            2: if (size) dx <= wb; else dx[ 7:0] <= wb[7:0];
            3: if (size) bx <= wb; else bx[ 7:0] <= wb[7:0];
            4: if (size) sp <= wb; else ax[15:8] <= wb[7:0];
            5: if (size) bp <= wb; else cx[15:8] <= wb[7:0];
            6: if (size) si <= wb; else dx[15:8] <= wb[7:0];
            7: if (size) di <= wb; else bx[15:8] <= wb[7:0];
            endcase

            t  <= next;
            cp <= 0;

        end
        // Либо записать в память младший байт wb
        else begin w <= 1; cp <= 1; m2 <= 1; o <= wb[7:0]; end

        // Дозапись старшего байта в память
        1: begin

            w    <= size;
            cp   <= size;
            m2   <= size;
            t    <= size ? WB : next;
            ea   <= ea + 1;
            o    <= wb[15:8];
            size <= 0;

        end

    endcase // Завершение m2

    // [3T] ВЫГРУЗКА WB -> В СТЕК
    PUSH: case (m3)
    0: begin cp <= 1; sp <= sp - 2; sgn <= ss;
             m3 <= 1; ea <= sp - 2; w <= 1; o <= wb[ 7:0]; end
    1: begin m3 <= 2; ea <= ea + 1; w <= 1; o <= wb[15:8]; end
    2: begin m3 <= 0; cp <= 0; t <= next; sgn <= ds; end
    endcase

    // [3T] ЗАГРУЗКА ИЗ СТЕКА -> WB
    POP: case (m3)
    0: begin m3 <= 1; cp <= 1; ea <= sp; sp <= sp + 2; sgn <= ss; end
    1: begin m3 <= 2; wb <= i; ea <= ea + 1; end
    2: begin m3 <= 0; wb[15:8] <= i; cp <= 0; t <= next; sgn <= ds; end
    endcase

    // [17T] ВЫЗОВ ПРОЦЕДУРЫ INTERRUPT
    INTERRUPT: case (m4)
    0: begin m4 <= 1; t <= PUSH; wb <= {4'hF, flags}; next <= INTERRUPT; end
    1: begin m4 <= 2; t <= PUSH; wb <= cs; end
    2: begin m4 <= 3; t <= PUSH; wb <= ip; end
    3: begin m4 <= 4; ea <= {interrupt, 2'b00}; sgn <= 0; cp <= 1; end
    4: begin m4 <= 5; ip[ 7:0] <= i; ea <= ea + 1; end
    5: begin m4 <= 6; ip[15:8] <= i; ea <= ea + 1; end
    6: begin m4 <= 7; cs[ 7:0] <= i; ea <= ea + 1; end
    7: begin m4 <= 0; cs[15:8] <= i; cp <= 0; t <= RUN; flags[IF] <= 1'b0; sgn <= ds; end
    endcase

    endcase

end

endmodule
